﻿' ===========================
'  AERIAL WARFARE — ADVANCED
' ===========================
' Controls:
'  Move  → Arrow Keys or WASD
'  Shoot → Space
'  Start → Enter
'  Pause → P
'
' New Enemy AI:
'   - Sine-Wave Flyers
'   - Diving Attackers
'   - Homing Missile Launchers
'
Option Strict On
Option Explicit On
Imports System.Drawing
Imports System.Windows.Forms
Imports System.Math

Public Class Form1
    Inherits Form

    ' GAME STATE -------------------------
    Private WithEvents gameTimer As Timer
    Private rand As New Random()

    Private player2 As Player
    Private bullets As New List(Of Bullet)
    Private enemies As New List(Of Enemy)
    Private missiles As New List(Of Missile)
    Private explosions As New List(Of Explosion)

    Private running As Boolean = False
    Private paused As Boolean = False

    Private score As Integer = 0
    Private level As Integer = 1
    Private spawnTimer As Integer = 0

    Public Sub New()
        Me.ClientSize = New Size(800, 600)
        Me.DoubleBuffered = True
        Me.KeyPreview = True
        Me.Text = "Aerial Warfare — Advanced AI"
        Me.StartPosition = FormStartPosition.CenterScreen
        Me.CenterToScreen()

        player2 = New Player(New PointF(ClientSize.Width \ 2, ClientSize.Height - 100))

        gameTimer = New Timer()
        gameTimer.Interval = 16

        AddHandler Me.KeyDown, AddressOf OnKeyDown
        AddHandler Me.KeyUp, AddressOf OnKeyUp
    End Sub

    ' ============================================
    '                   GAME FLOW
    ' ============================================
    Private Sub StartGame()
        running = True
        paused = False
        score = 0
        level = 1

        bullets.Clear()
        enemies.Clear()
        missiles.Clear()
        explosions.Clear()

        player2 = New Player(New PointF(ClientSize.Width \ 2, ClientSize.Height - 100))
        gameTimer.Start()
    End Sub

    Private Sub EndGame()
        running = False
        gameTimer.Stop()
    End Sub


    ' ============================================
    '                 INPUT HANDLING
    ' ============================================
    Private pressed As New HashSet(Of Keys)

    Private Sub OnKeyDown(sender As Object, e As KeyEventArgs)
        If e.KeyCode = Keys.Enter AndAlso Not running Then
            StartGame()
            Return
        End If

        If Not running Then Return

        If e.KeyCode = Keys.P Then
            paused = Not paused
            Return
        End If

        If e.KeyCode = Keys.Space Then
            Dim b = player2.Fire()
            If b IsNot Nothing Then bullets.Add(b)
        End If

        pressed.Add(e.KeyCode)
        UpdateMovement()
    End Sub

    Private Sub OnKeyUp(sender As Object, e As KeyEventArgs)
        If pressed.Contains(e.KeyCode) Then pressed.Remove(e.KeyCode)
        UpdateMovement()
    End Sub

    Private Sub UpdateMovement()
        Dim x As Single = 0
        Dim y As Single = 0

        If pressed.Contains(Keys.Left) Or pressed.Contains(Keys.A) Then x -= 1
        If pressed.Contains(Keys.Right) Or pressed.Contains(Keys.D) Then x += 1
        If pressed.Contains(Keys.Up) Or pressed.Contains(Keys.W) Then y -= 1
        If pressed.Contains(Keys.Down) Or pressed.Contains(Keys.S) Then y += 1

        player2.SetDirection(x, y)
    End Sub


    ' ============================================
    '                MAIN GAME LOOP
    ' ============================================
    Private Sub gameTimer_Tick(sender As Object, e As EventArgs) Handles gameTimer.Tick
        If Not running Or paused Then Invalidate() : Return

        player2.Update(ClientSize)

        ' ---- SPAWN ENEMIES ----
        spawnTimer += 1
        If spawnTimer >= Math.Max(25 - level * 2, 10) Then
            spawnTimer = 0
            SpawnAdvancedEnemy()
        End If

        ' ---- UPDATE BULLETS ----
        For i = bullets.Count - 1 To 0 Step -1
            bullets(i).Update()
            If bullets(i).Position.Y < -30 Then bullets.RemoveAt(i)
        Next

        ' ---- UPDATE ENEMIES ----
        For i = enemies.Count - 1 To 0 Step -1
            enemies(i).Update(player2.Position)

            ' Heavy AI can fire missiles
            Dim m = enemies(i).TryFireMissile(player2.Position)
            If m IsNot Nothing Then missiles.Add(m)

            If enemies(i).OffScreen(ClientSize) Then enemies.RemoveAt(i)
        Next

        ' ---- UPDATE MISSILES ----
        For i = missiles.Count - 1 To 0 Step -1
            missiles(i).Update(player2.Position)
            If missiles(i).OffScreen(ClientSize) Then missiles.RemoveAt(i)
        Next

        ' ---- UPDATE EXPLOSIONS ----
        For i = explosions.Count - 1 To 0 Step -1
            explosions(i).Update()
            If explosions(i).Finished Then explosions.RemoveAt(i)
        Next

        ' ---- COLLISIONS ----
        HandleCollisions()

        Invalidate()
    End Sub


    ' ============================================
    '               COLLISION HANDLING
    ' ============================================
    Private Sub HandleCollisions()

        ' Player vs Missiles
        For i = missiles.Count - 1 To 0 Step -1
            If player2.Bounds.IntersectsWith(missiles(i).Bounds) Then
                missiles.RemoveAt(i)
                player2.Lives -= 1
                explosions.Add(New Explosion(player2.Position))
                If player2.Lives <= 0 Then EndGame()
            End If
        Next

        ' Player vs Enemies
        For i = enemies.Count - 1 To 0 Step -1
            If player2.Bounds.IntersectsWith(enemies(i).Bounds) Then
                explosions.Add(New Explosion(player2.Position))
                player2.Lives -= 1
                enemies.RemoveAt(i)
                If player2.Lives <= 0 Then EndGame()
            End If
        Next

        ' Bullets vs Enemies
        For b = bullets.Count - 1 To 0 Step -1
            For en = enemies.Count - 1 To 0 Step -1
                If bullets(b).Bounds.IntersectsWith(enemies(en).Bounds) Then

                    enemies(en).Health -= bullets(b).Damage
                    bullets.RemoveAt(b)

                    If enemies(en).Health <= 0 Then
                        score += enemies(en).ScoreValue
                        explosions.Add(New Explosion(enemies(en).Position))
                        enemies.RemoveAt(en)
                        If score >= level * 150 Then level += 1
                    End If

                    Exit For
                End If
            Next
        Next
    End Sub


    ' ============================================
    '              ENEMY SPAWNING (ADVANCED)
    ' ============================================
    Private Sub SpawnAdvancedEnemy()
        Dim x = rand.Next(40, ClientSize.Width - 40)
        Dim pos As New PointF(x, -50)
        Dim roll = rand.NextDouble()

        Dim enemy As Enemy

        If roll < 0.45 Then
            enemy = enemy.CreateSineFlyer(pos, level)
        ElseIf roll < 0.75 Then
            enemy = Enemy.CreateDiver(pos, level)
        Else
            enemy = Enemy.CreateMissileLauncher(pos, level)
        End If

        enemies.Add(enemy)
    End Sub


    ' ============================================
    '                   DRAWING
    ' ============================================
    Protected Overrides Sub OnPaint(e As PaintEventArgs)
        Dim g = e.Graphics
        g.SmoothingMode = Drawing2D.SmoothingMode.AntiAlias

        ' Background
        Using b As New Drawing2D.LinearGradientBrush(ClientRectangle,
                                                     Color.SkyBlue,
                                                     Color.SteelBlue,
                                                     90.0F)
            g.FillRectangle(b, ClientRectangle)
        End Using

        ' Player
        If running Then player2.Draw(g)

        ' Bullets
        For Each b In bullets : b.Draw(g) : Next

        ' Enemies
        For Each en In enemies : en.Draw(g) : Next

        ' Missiles
        For Each m In missiles : m.Draw(g) : Next

        ' Explosions
        For Each ex In explosions : ex.Draw(g) : Next

        ' HUD
        Using f As New Font("Segoe UI", 12, FontStyle.Bold)
            g.DrawString($"Score: {score}", f, Brushes.White, 10, 10)
            g.DrawString($"Lives: {player2.Lives}", f, Brushes.White, 10, 30)
            g.DrawString($"Level: {level}", f, Brushes.White, 10, 50)
        End Using

        If Not running Then
            DrawCentered(g, "AERIAL WARFARE — ADVANCED", 28, -40)
            DrawCentered(g, "Press ENTER to Start", 18, 10)
            DrawCentered(g, "Move: Arrows or WASD  •  Shoot: Space  •  Pause: P", 12, 40)
        ElseIf paused Then
            DrawCentered(g, "PAUSED", 32, -10)
        End If

        MyBase.OnPaint(e)
    End Sub

    Private Sub DrawCentered(g As Graphics, text As String, size As Integer, offsetY As Integer)
        Using f As New Font("Segoe UI", size, FontStyle.Bold)
            Dim s = g.MeasureString(text, f)
            g.DrawString(text, f, Brushes.White,
                         (ClientSize.Width - s.Width) / 2,
                         (ClientSize.Height - s.Height) / 2 + offsetY)
        End Using
    End Sub


    ' ===================================================================
    ' ===============  CLASSES BELOW (PLAYER / ENEMIES) ==================
    ' ===================================================================

    ' -----------------------------------------------------------
    '  PLAYER
    ' -----------------------------------------------------------
    Private Class Player
        Public Position As PointF
        Public Lives As Integer = 5
        Private speed As Single = 5.5F
        Private vx As Single = 0
        Private vy As Single = 0
        Private fireCD As Integer = 0

        Public Sub New(pos As PointF)
            Position = pos
        End Sub

        Public Sub SetDirection(x As Single, y As Single)
            If x <> 0 Or y <> 0 Then
                Dim L = CSng(Sqrt(x * x + y * y))
                x /= L
                y /= L
            End If
            vx = x : vy = y
        End Sub

        Public Sub Update(bounds As Size)
            Position = New PointF(
                Position.X + vx * speed,
                Position.Y + vy * speed
            )

            ' Clamp
            Position = New PointF(
                Max(20, Min(bounds.Width - 20, Position.X)),
                Max(20, Min(bounds.Height - 20, Position.Y))
            )

            If fireCD > 0 Then fireCD -= 1
        End Sub

        Public Function Fire() As Bullet
            If fireCD > 0 Then Return Nothing
            fireCD = 8
            Return New Bullet(New PointF(Position.X, Position.Y - 24),
                              New PointF(0, -11), 7)
        End Function

        Public ReadOnly Property Bounds As RectangleF
            Get
                Return New RectangleF(Position.X - 18, Position.Y - 18, 36, 36)
            End Get
        End Property

        Public Sub Draw(g As Graphics)
            Dim p1 = New PointF(Position.X, Position.Y - 16)
            Dim p2 = New PointF(Position.X - 16, Position.Y + 12)
            Dim p3 = New PointF(Position.X + 16, Position.Y + 12)
            g.FillPolygon(Brushes.Yellow, {p1, p2, p3})
            g.DrawPolygon(Pens.Black, {p1, p2, p3})
        End Sub
    End Class


    ' -----------------------------------------------------------
    '  BULLET
    ' -----------------------------------------------------------
    Private Class Bullet
        Public Position As PointF
        Public Velocity As PointF
        Public Damage As Integer

        Public Sub New(p As PointF, v As PointF, d As Integer)
            Position = p
            Velocity = v
            Damage = d
        End Sub

        Public Sub Update()
            Position = New PointF(Position.X + Velocity.X,
                                  Position.Y + Velocity.Y)
        End Sub

        Public ReadOnly Property Bounds As RectangleF
            Get
                Return New RectangleF(Position.X - 2, Position.Y - 8, 4, 14)
            End Get
        End Property

        Public Sub Draw(g As Graphics)
            g.FillRectangle(Brushes.White, Position.X - 2, Position.Y - 8, 4, 14)
        End Sub
    End Class


    ' -----------------------------------------------------------
    '  MISSILE (HOMING)
    ' -----------------------------------------------------------
    Private Class Missile
        Public Position As PointF
        Private angle As Single
        Private speed As Single = 4
        Private turnRate As Single = 0.08F ' rad/frame

        Public Sub New(pos As PointF, playerPos As PointF)
            Position = pos
            angle = CSng(Atan2(playerPos.Y - pos.Y, playerPos.X - pos.X))
        End Sub

        Public Sub Update(playerPos As PointF)
            Dim targetA = Atan2(playerPos.Y - Position.Y,
                                 playerPos.X - Position.X)
            Dim diff = NormalizeAngle(CSng(targetA - angle))

            If Abs(diff) < turnRate Then
                angle = CSng(targetA)
            ElseIf diff > 0 Then
                angle += turnRate
            Else
                angle -= turnRate
            End If

            Position = New PointF(
                CSng(Position.X + Cos(angle) * speed),
                CSng(Position.Y + Sin(angle) * speed)
            )
        End Sub

        Private Function NormalizeAngle(a As Single) As Single
            While a > PI : a -= CSng(2 * PI) : End While
            While a < -PI : a += CSng(2 * PI) : End While
            Return a
        End Function

        Public Function OffScreen(sz As Size) As Boolean
            Return Position.X < -50 Or Position.X > sz.Width + 50 Or
                   Position.Y < -50 Or Position.Y > sz.Height + 50
        End Function

        Public ReadOnly Property Bounds As RectangleF
            Get
                Return New RectangleF(Position.X - 5, Position.Y - 5, 10, 10)
            End Get
        End Property

        Public Sub Draw(g As Graphics)
            g.FillEllipse(Brushes.Red, Bounds)
            g.DrawEllipse(Pens.Black, Bounds)
        End Sub
    End Class


    ' -----------------------------------------------------------
    '  ENEMY (Advanced AI Varieties)
    ' -----------------------------------------------------------
    Private Class Enemy

        Public Enum AIType
            SineFlyer
            Diver
            MissileLauncher
        End Enum

        Public Position As PointF
        Public Velocity As PointF
        Public Type As AIType

        Public Health As Integer
        Public ScoreValue As Integer
        Private level As Integer

        ' For sine movement
        Private sineCounter As Single = 0
        Private sineSpeed As Single

        ' Missile launcher cooldown
        Private missileCD As Integer = 0

        Private Sub New(pos As PointF, t As AIType, lvl As Integer)
            Position = pos
            Type = t
            level = lvl
        End Sub

        ' Factory: Sine flyer
        Public Shared Function CreateSineFlyer(pos As PointF, lvl As Integer) As Enemy
            Dim e As New Enemy(pos, AIType.SineFlyer, lvl)
            e.Health = 12 + lvl * 2
            e.ScoreValue = 15
            e.Velocity = New PointF(0, 2.5F + lvl * 0.1F)
            e.sineSpeed = 0.14F + lvl * 0.02F
            Return e
        End Function

        ' Factory: Diver
        Public Shared Function CreateDiver(pos As PointF, lvl As Integer) As Enemy
            Dim e As New Enemy(pos, AIType.Diver, lvl)
            e.Health = 18 + lvl * 3
            e.ScoreValue = 30
            e.Velocity = New PointF(0, 2.8F + lvl * 0.15F)
            Return e
        End Function

        ' Factory: Missile launcher
        Public Shared Function CreateMissileLauncher(pos As PointF, lvl As Integer) As Enemy
            Dim e As New Enemy(pos, AIType.MissileLauncher, lvl)
            e.Health = 35 + lvl * 5
            e.ScoreValue = 60
            e.Velocity = New PointF(0, 1.7F + lvl * 0.05F)
            Return e
        End Function

        Public Sub Update(playerPos As PointF)
            Select Case Type

                Case AIType.SineFlyer
                    sineCounter += sineSpeed
                    Position = New PointF(
                        Position.X + CSng(Sin(sineCounter) * 4),
                        Position.Y + Velocity.Y
                    )

                Case AIType.Diver
                    If Position.Y > 150 Then
                        ' home in on player horizontally
                        Dim dx = playerPos.X - Position.X
                        Position = New PointF(
                            Position.X + Math.Sign(dx) * 3.2F,
                            Position.Y + Velocity.Y + 0.4F
                        )
                    Else
                        Position = New PointF(
                            Position.X,
                            Position.Y + Velocity.Y
                        )
                    End If

                Case AIType.MissileLauncher
                    Position = New PointF(Position.X, Position.Y + Velocity.Y)
                    If missileCD > 0 Then missileCD -= 1
            End Select
        End Sub

        Public Function TryFireMissile(playerPos As PointF) As Missile
            If Type <> AIType.MissileLauncher Then Return Nothing

            If missileCD <= 0 AndAlso Position.Y > 80 Then
                missileCD = 75
                Return New Missile(New PointF(Position.X, Position.Y + 20),
                                   playerPos)
            End If

            Return Nothing
        End Function

        Public Function OffScreen(sz As Size) As Boolean
            Return Position.Y > sz.Height + 50 Or
                   Position.X < -100 Or Position.X > sz.Width + 100
        End Function

        Public ReadOnly Property Bounds As RectangleF
            Get
                Dim s As Integer = If(Type = AIType.MissileLauncher, 48,
                         If(Type = AIType.Diver, 36, 28))
                Return New RectangleF(CSng(Position.X - s / 2), CSng(Position.Y - s / 2), s, s)
            End Get
        End Property

        Public Sub Draw(g As Graphics)
            Dim col As Brush = Brushes.OrangeRed
            If Type = AIType.SineFlyer Then col = Brushes.Gold
            If Type = AIType.Diver Then col = Brushes.DarkRed
            If Type = AIType.MissileLauncher Then col = Brushes.Purple

            g.FillEllipse(col, Bounds)
            g.DrawEllipse(Pens.Black, Bounds)
        End Sub
    End Class


    ' -----------------------------------------------------------
    '  EXPLOSION
    ' -----------------------------------------------------------
    Private Class Explosion
        Public Position As PointF
        Private t As Integer = 0
        Private maxT As Integer = 20

        Public Sub New(pos As PointF)
            Position = pos
        End Sub

        Public Sub Update()
            t += 1
        End Sub

        Public ReadOnly Property Finished As Boolean
            Get
                Return t >= maxT
            End Get
        End Property

        Public Sub Draw(g As Graphics)
            Dim k = t / CSng(maxT)
            Dim size = 20 + 50 * k
            Dim alpha = CInt(255 * (1 - k))
            Using br As New SolidBrush(Color.FromArgb(alpha, Color.Orange))
                g.FillEllipse(br,
                              Position.X - size / 2,
                              Position.Y - size / 2,
                              size, size)
            End Using
        End Sub
    End Class

    ' -----------------------------------------------------------
    '   PROGRAM ENTRYPOINT
    ' -----------------------------------------------------------
    <STAThread>
    Public Shared Sub Main()
        Application.EnableVisualStyles()
        Application.Run(New Form1())
    End Sub

End Class

